#include <htc.h>
volatile unsigned short CCPR1 @ 0x291;
#define _XTAL_FREQ 32000000
__CONFIG(FOSC_INTOSC & WDTE_OFF & PWRTE_OFF & MCLRE_OFF & BOREN_OFF & CP_OFF & CPD_OFF & CLKOUTEN_OFF & IESO_OFF & FCMEN_OFF);
__CONFIG(WRT_OFF & PLLEN_OFF & PLLEN_OFF & BORV_19 & LVP_OFF);
__IDLOC(0000);
typedef unsigned char UInt8;
typedef unsigned short UInt16;
typedef unsigned long UInt32;
typedef signed char Int8;
typedef signed short Int16;
typedef signed long Int32;
#define NULL ((void*)0)


//versioning
	static const UInt8 gSwVersion = 3;

//battery voltage
	static UInt16 gBattCentiVolts = 0;

//PWM control
	static UInt8 gLed[4] = {0,0,0,0};
	static UInt16 gPwmCtr = 0;

//input data
	static UInt8 gRxData[32];
	static UInt8 gRxPos;
	static UInt8 gRxMask;
	static bit gRxDone;

//data for int
	static UInt16 gIntCapturedTime;
	static bit gIntFirstEdgeSeen;

UInt8 eeRead(UInt8 addr){

	EECON1 = 0b00000000;	//read data
	EEADRL = addr;
	EECON1bits.RD = 1;		//do it
	return EEDATL;
}

void eeWrite(UInt8 addr, UInt8 data){

	static bit gie;

	EECON1= 0b00000100;	//write data
	EEADRL = addr;
	EEDATL = data;
	gie = GIE;
	INTCONbits.GIE = 0;
	EECON2 = 0x55;
	EECON2 = 0xAA;
	EECON1bits.WR = 1;
	INTCONbits.GIE = gie;
	while(EECON1bits.WR);
}

void log(UInt8 a){

	static UInt16 addr = 0;

	if(addr < 0x100) eeWrite(addr++, a);
}

void rxStart(void){

	UInt8 i;

	for(i = 0; i < sizeof(gRxData); i++) gRxData[i] = 0;
	gRxPos = 0;
	gRxMask = 0x80;
	gRxDone = 0;

	gIntFirstEdgeSeen = 0;
	
	CCP1IF = 0;
	CCP1IE = 1;
	TMR2ON = 0;
	TMR2IE = 1;
	TMR2IF = 0;
	CCP1CON = 0b00000100;	//catch falling edge (IR turns on)
}

UInt8 rxDecode(void){		//0 if success, else error

	UInt8 i, j, t, k;


	if(gRxPos < 14) return 1;			//size error
	if(gRxData[0] != 0xFF) return 2;	//preamble error
	if(gRxData[1] != 0x05) return 3;	//header error

	//check crc
	for(k = 0, i = 0; i < gRxPos - 3; i++){
		
		t = gRxData[i + 2];
		for(j = 0; j < 8; j++, t <<= 1){

			k = (k << 1) ^ (((t ^ k) & 0x80) ? 0x83 : 0);
		}
	}

	if(k != gRxData[gRxPos - 1]) return 4;	//crc error

	return 0;
}

static UInt32 getU32(UInt8* t){

	UInt32 v = t[3];

	v = (v << 8) | t[2];
	v = (v << 8) | t[1];
	v = (v << 8) | t[0];

	return v;
}

static UInt16 getU16(UInt8* t){

	UInt32 v = t[1];

	v = (v << 8) | t[0];

	return v;
}

static UInt8 getU8(UInt8* t){

	return t[0];
}

//offsets for main data
#define OFF_MAGIC			0xFF	//EEPROM offsets
#define OFF_MODE			0xFE


//offsets for OFF mode
#define OFF_OFF_PREV_MODE	0xCF

//offsets for SOLID mode
#define OFF_SOLID_B			0xC8
#define OFF_SOLID_G			0xC7
#define OFF_SOLID_R			0xC6
#define OFF_SOLID_W			0xC5

//offsets for FADE mode
#define OFF_FADE_SPEED		0xBF
#define OFF_FADE_BRI		0xBE


#define MAGIC		'M'		//main magic val




#define MODE_OFF		0	//modes
#define MODE_SOLID		1
#define MODE_FADE		2
#define MODE_MAX		3

#define FPTR_INIT		0
#define FPTR_DESTROY	1
#define FPTR_BTN_UP		2
#define FPTR_BTN_DOWN	3
#define FPTR_BTN_O		4
#define FPTR_DATA_INIT	5
#define FPTR_RUN		6
#define FPTR_MAX		7

union{
	struct{

		UInt8 selector;

	}solid;

	struct{
		struct{

			UInt8 val : 5;
			UInt8 changing : 1;
			UInt8 rising: 1;
		}tmp[4];
		UInt8 speed:7;
		UInt8 adjBri:1;
		UInt8 bri;
		UInt16 speedLeft;

	}fade;

	struct{

		UInt16 configModeStatus;
	}off;

}tempData;

UInt8 exponents[32];

#define log math_log
#include <math.h>
#include <float.h>
#undef math_log

static UInt16 gRngSeed;
/*
	config mode is entered from off mode by pressing buttons
	up = 00 down = 01 O = 10. lower bits are earlier preses
	11 = end of sequence
*/
static const UInt16 gConfigSteps = 0b11101001000100;	//UDUDOO


static UInt8 rand(void){

	static UInt8 ctr = 0;
	
	gRngSeed = gRngSeed * 0x5EED + 13;
	return gRngSeed >> 8;
}

void delay_ms(UInt16 ms){		//uses TMR1 so it accounts for all interrupts' lost time

	UInt16 d, t;
	
	while(ms){

		d = (ms > 50) ? 50 : ms;
		ms -= d;
		d *= 1000;
		t = TMR1;
		while(TMR1 - t < d);
	}
}

static void genExponents(UInt8 bri){

	UInt8 i;
	double t;

	for(i = 0; i < 32; i++){

		t = i;
		t = exp(t / 9);
		t -= 1;
		t *= 8.4;
		t *= bri;
		t /= 255;
		t += 0.5;	//round

		exponents[i] = (UInt8)t;
	}
}

UInt8 mOffInit(UInt8 arg){

	if(arg != MODE_OFF){	//save previous mode

		eeWrite(OFF_OFF_PREV_MODE, arg);
	}
	gLed[0] = 0;
	gLed[1] = 0;
	gLed[2] = 0;
	gLed[3] = 0;
	tempData.off.configModeStatus = gConfigSteps;
	return 0;
}

UInt8 mOffDestr(UInt8 arg){

	return eeRead(OFF_OFF_PREV_MODE);	//return old mode
}

UInt8 mOffDataInit(UInt8 arg){

	eeWrite(OFF_OFF_PREV_MODE, MODE_FADE);

	return 0;
}

static void cfgModeShowVal(UInt8 colorMarker, UInt32 v){

	//marker
	{
		gLed[0] = 0;
		gLed[1] = (colorMarker & 4) ? 100 : 0;
		gLed[2] = (colorMarker & 2) ? 100 : 0;
		gLed[3] = (colorMarker & 1) ? 100 : 0;
		delay_ms(1000);
		gLed[1] = 0;
		gLed[2] = 0;
		gLed[3] = 0;
	}

	//number itself, one decimal digit at a time, in proper order
	{
		UInt32 t = v;
		UInt8 digit = 0;

		//count digits
		while(t){
			digit++;
			t /= 10;
		}
		if(!digit) digit = 1;	//ero is still one digit

		//show them
		t = 1;
		while(--digit) t *= 10;

		do{

			digit = v / t;
			v %= t;
			t /= 10;

			while(digit--){
				delay_ms(300);	//wait
				gLed[0] = 10;
				delay_ms(500);	//blink
				gLed[0] = 0;
			}
				
			delay_ms(300);	//wait
			gLed[0] = 255;
			delay_ms(700);	//show end of digit
			gLed[0] = 0;
			delay_ms(500);	//wait

		}while(t);
	}	
}

static void mOffCheckConfigMode(UInt8 btn){

	UInt8 t = (tempData.off.configModeStatus & 3) ^ btn;

	tempData.off.configModeStatus >>= 2;
	if(!t && (tempData.off.configModeStatus & 3) == 0b11){		//right code and we're done

		//config mode
		{
			gLed[0] = 0;
			gLed[1] = 0;
			gLed[2] = 0;
			gLed[3] = 0;
			cfgModeShowVal(0b100, gSwVersion);		//red -> sw version
			cfgModeShowVal(0b010, gBattCentiVolts);	//green -> batt voltage
			gLed[0] = 0;
			gLed[1] = 0;
			gLed[2] = 0;
			gLed[3] = 0;
		}
		tempData.off.configModeStatus = gConfigSteps;
	}
	else if(t) tempData.off.configModeStatus = gConfigSteps;	//wrong code
}

UInt8 mOffUp(UInt8 arg){

	mOffCheckConfigMode(0b00);
	return 0;
}

UInt8 mOffDown(UInt8 arg){

	mOffCheckConfigMode(0b01);
	return 0;
}

UInt8 mOffO(UInt8 arg){

	mOffCheckConfigMode(0b10);
	return 0;
}

UInt8 mSolidInit(UInt8 arg){

	gLed[0] = eeRead(OFF_SOLID_W);
	gLed[1] = eeRead(OFF_SOLID_R);
	gLed[2] = eeRead(OFF_SOLID_G);
	gLed[3] = eeRead(OFF_SOLID_B);

	tempData.solid.selector = 0;

	return 0;
}

UInt8 mSolidDestr(UInt8 arg){

	eeWrite(OFF_SOLID_W, gLed[0]);
	eeWrite(OFF_SOLID_R, gLed[1] );
	eeWrite(OFF_SOLID_G, gLed[2]);
	eeWrite(OFF_SOLID_B, gLed[3]);

	return 0;
}

UInt8 mSolidUp(UInt8 arg){

	UInt16 t = gLed[tempData.solid.selector];

	t += 1 + (t + 3) / 4;
	if(t > 255) t = 255;
	gLed[tempData.solid.selector] = t;

	return 0;
}

UInt8 mSolidDown(UInt8 arg){

	Int16 t = gLed[tempData.solid.selector];

	t -= 1 + (t + 3) / 4;
	if(t < 0) t = 0;
	gLed[tempData.solid.selector] = t;

	return 0;
}

UInt8 mSolidO(UInt8 arg){

	UInt8 s0, s1, s2, s3;

	if(++tempData.solid.selector == 4) tempData.solid.selector = 0;
	s0 = gLed[0];
	s1 = gLed[1];
	s2 = gLed[2];
	s3 = gLed[3];
	gLed[0] = 0;
	gLed[1] = 0;
	gLed[2] = 0;
	gLed[3] = 0;
	delay_ms(100);
	gLed[tempData.solid.selector] = 100;
	delay_ms(500);
	gLed[tempData.solid.selector] = 0;
	delay_ms(100);
	gLed[0] = s0;
	gLed[1] = s1;
	gLed[2] = s2;
	gLed[3] = s3;

	return 0;
}

UInt8 mSolidDataInit(UInt8 arg){

	eeWrite(OFF_SOLID_W, 5);
	eeWrite(OFF_SOLID_R, 5);
	eeWrite(OFF_SOLID_G, 5);
	eeWrite(OFF_SOLID_B, 100);

	return 0;
}

UInt8 mFadeInit(UInt8 arg){

	UInt8 i;

	gLed[0] = 0;
	gLed[1] = 0;
	gLed[2] = 0;
	gLed[3] = 0;
	tempData.fade.speed = eeRead(OFF_FADE_SPEED);
	tempData.fade.bri = eeRead(OFF_FADE_BRI);
	tempData.fade.speedLeft = 1;	//start soon
	tempData.fade.adjBri = 0;
	for(i = 0; i < 4; i++){

		tempData.fade.tmp[i].val = 0;
		tempData.fade.tmp[i].changing = 0;
	}
	genExponents(tempData.fade.bri);

	return 0;
}

UInt8 mFadeDestr(UInt8 arg){

	eeWrite(OFF_FADE_SPEED, tempData.fade.speed);
	eeWrite(OFF_FADE_BRI, tempData.fade.bri);

	return 0;
}

UInt8 mFadeUp(UInt8 arg){

	UInt16 t;

	if(tempData.fade.adjBri){

		t = (((UInt16)tempData.fade.bri)*5)/4;
		tempData.fade.bri = (t > 255) ? 255 : t;
		genExponents(tempData.fade.bri);
	}
	else{

		if(tempData.fade.speed > 2) tempData.fade.speed--;
	}
	return 0;
}

UInt8 mFadeDown(UInt8 arg){

	UInt8 t;

	if(tempData.fade.adjBri){

		t = (((UInt16)tempData.fade.bri)*4)/5;
		tempData.fade.bri = (t < 5) ? 5 : t;
		genExponents(tempData.fade.bri);
	}
	else{

		if(tempData.fade.speed < 40) tempData.fade.speed++;

	}
	return 0;
}

UInt8 mFadeO(UInt8 arg){

	UInt8 s0, s1, s2, s3;

	s0 = gLed[0];
	s1 = gLed[1];
	s2 = gLed[2];
	s3 = gLed[3];
	
	if(tempData.fade.adjBri){	//switch to adjusting speed

		gLed[0] = 100;
		gLed[1] = 0;
		gLed[2] = 0;
		gLed[3] = 0;
		delay_ms(150);
		gLed[0] = 0;
		gLed[1] = 100;
		delay_ms(150);
		gLed[1] = 0;
		gLed[2] = 100;
		delay_ms(150);
		gLed[2] = 0;
		gLed[3] = 100;
		delay_ms(150);
		tempData.fade.adjBri = 0;
	}
	else{						//switch to adjusting brightness

		gLed[0] = 0;
		gLed[1] = 0;
		gLed[2] = 0;
		gLed[3] = 0;
		delay_ms(50);
		gLed[0] = 1;
		delay_ms(100);
		gLed[0] = 0;
		delay_ms(50);
		gLed[0] = 30;
		delay_ms(100);
		gLed[0] = 0;
		delay_ms(50);
		gLed[0] = 80;
		delay_ms(100);
		gLed[0] = 0;
		delay_ms(50);
		gLed[0] = 200;
		delay_ms(100);
		gLed[0] = 0;
		delay_ms(50);
		tempData.fade.adjBri = 1;
	}

	gLed[0] = s0;
	gLed[1] = s1;
	gLed[2] = s2;
	gLed[3] = s3;

	return 0;
}

UInt8 mFadeDataInit(UInt8 arg){

	eeWrite(OFF_FADE_SPEED, 10);
	eeWrite(OFF_FADE_BRI, 255);
	
	return 0;
}

#define chance(__val) (rand() < (__val))

UInt8 mFadeRun(UInt8 arg){

	UInt8 i, t;

	if(!--tempData.fade.speedLeft){

		tempData.fade.speedLeft = ((UInt16)tempData.fade.speed) << 7;

		for(i = 0; i < 4; i++){
			
			//first update based on current rise/fall, fancel them if we hit boundaries
			if(tempData.fade.tmp[i].changing){
				t = tempData.fade.tmp[i].val;
				if(tempData.fade.tmp[i].rising){
					if(t == 0x1F){
						tempData.fade.tmp[i].changing = 0;
					}
					else{
						t++;
					}
				}
				else{
					if(t == 0){
						tempData.fade.tmp[i].changing = 0;
					}
					else{
						t--;
					}
				}
				tempData.fade.tmp[i].val = t;
			}

			//then decide what to do probabilistically
			if(tempData.fade.tmp[i].changing && chance((0x1F - tempData.fade.tmp[i].val) / 4 + 1)){	//chance of stopping a rise or a fall is very small and lessens the higher the value

				tempData.fade.tmp[i].changing = 0;
			}
			else if(!tempData.fade.tmp[i].changing && chance(8)){									//chance of starting a rise or a fall

				tempData.fade.tmp[i].changing = 1;
				tempData.fade.tmp[i].rising = !chance(tempData.fade.tmp[i].val << 3);				//thehigher the value, the less likely the fall and not the rise
			}
		}

		gLed[0] = exponents[tempData.fade.tmp[0].val];			//white is special (we dim it)
		if(gLed[0] > 32) gLed[0] = ((gLed[0] - 32) >> 1) + 32;
		for(i = 1; i < 4; i++) gLed[i] = exponents[tempData.fade.tmp[i].val];
	}

	return 0;
}

typedef UInt8 (*modeFunc)(UInt8 arg);

static const modeFunc gModeFns[MODE_MAX][FPTR_MAX] =	{
															{mOffInit, mOffDestr, mOffUp, mOffDown, mOffO, mOffDataInit, NULL},
															{mSolidInit, mSolidDestr, mSolidUp, mSolidDown, mSolidO, mSolidDataInit, NULL},
															{mFadeInit, mFadeDestr, mFadeUp, mFadeDown, mFadeO, mFadeDataInit, mFadeRun}
														};											

void init(void){
	PORTA		= 0b00000000;	// output low
	ANSELA		= 0b00000000;	// no analog pins
	TRISA		= 0b11101000;	// all in, RA0,1,2,4 out
	OSCCON		= 0b11110000;	// 32MHz
	OPTION_REG	= 0b10001000;	// pullups off, Tmr0 @ Fosc/8
	INTCON		= 0b10100000;	// ints on, TMR0 used for PWM
	APFCON		= 0b00000001;	// CCP1 is on RA5
	T1CON		= 0b00110001;	// TMR1 at Fosc/32 = 1 MHz
	T2CON		= 0b00001011;	// TMR2 at Fosc/256 = 125 KHz, overflow interrupts dithered by 2, so overflowing at 244.14 Hz (every 4 ms or so)

	//delay till everything is ready
	__delay_ms(300);
	while(!HFIOFL || !PLLR);
}

static UInt8 runFuncPtr(UInt8 funcIdx, UInt8 curMode, UInt8 arg){

	modeFunc f = gModeFns[curMode][funcIdx];

	return f ? f(arg) : 0;
}

void main(void){

	UInt8 mode;

	init();

	//save/restore values
	if(eeRead(OFF_MAGIC) == MAGIC){

		mode = eeRead(OFF_MODE);
	}
	else{

		for(mode = 0; mode < MODE_MAX; mode++) runFuncPtr(FPTR_DATA_INIT, mode, 0);
		eeWrite(OFF_MAGIC, MAGIC);
		eeWrite(OFF_MODE, mode = MODE_OFF);
	}


	rxStart();						//start initial receive
	runFuncPtr(FPTR_INIT, mode, 0);	//init this mode

	//main loop
	while(1){
		UInt8 t;
		UInt16 pt = 0;

		if(pt != TMR1 >> 6){	//run at most once every 64ms
			runFuncPtr(FPTR_RUN, mode, 0);
			pt = TMR1 >> 6;
		}
		if(gRxDone){

			t = rxDecode();
			if(t == 0){
				UInt32 pressesTotal = getU32(gRxData + 2);
				UInt32 pressessSinceBattChange = getU32(gRxData + 6);
				UInt8 whichBtn = getU8(gRxData + 12);
		
				gBattCentiVolts = getU16(gRxData + 10);
				if(whichBtn == 1){			//Mode
	
					if(mode != MODE_OFF){		//mode button is ignored when off
	
						runFuncPtr(FPTR_DESTROY, mode, 0);
						if(++mode == MODE_MAX) mode = 1;
						runFuncPtr(FPTR_INIT, mode, 0);
						eeWrite(OFF_MODE, mode);
					}
				}
				else if(whichBtn == 4){		//UP
			
					runFuncPtr(FPTR_BTN_UP, mode, 0);
				}
				else if(whichBtn == 3){		//DOWN
	
					runFuncPtr(FPTR_BTN_DOWN, mode, 0);
				}
				else if(whichBtn == 2){		//"O" = enter
	
					runFuncPtr(FPTR_BTN_O, mode, 0);
				}
				else if(whichBtn == 0){		//On-Off
	
					if(mode == MODE_OFF){

						mode = runFuncPtr(FPTR_DESTROY, mode, 0);	//returns new mode
						runFuncPtr(FPTR_INIT, mode, 0);
					}
					else{

						runFuncPtr(FPTR_DESTROY, mode, 0);
						runFuncPtr(FPTR_INIT, MODE_OFF, mode);
						mode = MODE_OFF;
					}
				}
			}
			rxStart();
		}
	}

	while(1);
}

static void interrupt isr(void){

	UInt8 t;
	static bit v = 0;

	if(TMR0IF){
		TMR0IF = 0;
		gPwmCtr++;
		gPwmCtr &=~ 0x0200;
		t = LATA &~ 0x17;
		if((gPwmCtr<<1) < (UInt16)gLed[0]) t |= 0x01;	//white always dimmer
		if((gPwmCtr>>1) < gLed[1]) t |= 0x10;	//red always brighter
		if(gPwmCtr < gLed[2]) t |= 0x02;
		if(gPwmCtr < gLed[3]) t |= 0x04;
		LATA = t;
	}
	if(TMR2IF){		//timeout

		TMR2IF	= 0;

		TMR2ON = 0;
		gRxDone = 1;
		CCP1CON = 0;
		CCP1IF = 0;
	}
	if(CCP1IF){
	
		CCP1IF	= 0;	//clear this flag
	
		if(gIntFirstEdgeSeen){		//rx going on

			if(CCP1CON & 1){	//begin timed period

				gIntCapturedTime = CCPR1;
				TMR2 = 0;
			}
			else{				//end timed perioud

				gIntCapturedTime = CCPR1 - gIntCapturedTime;

				if(gIntCapturedTime > 1150ul) gRxData[gRxPos] |= gRxMask;	//one
				gRxMask >>= 1;
				if(!gRxMask){
					gRxMask = 0x80;
					gRxPos++;
		
					if(gRxPos == sizeof(gRxData)){
						CCP1CON = 0;
						gRxDone = 1;
					}
				}
			}
		}
		else{						//rx just stared

			gIntFirstEdgeSeen = 1;
			TMR2ON = 1;
			TMR2 = 0;
			TMR2IE = 1;
			TMR2IF = 0;
		}
		if(!gRxDone) CCP1CON ^= 1;		//catch other edge
	}
}